<?php

namespace Yoast\WP\SEO\Premium\Integrations;

use WP_Admin_Bar;
use WPSEO_Metabox_Analysis_Readability;
use WPSEO_Metabox_Analysis_SEO;
use Yoast\WP\SEO\Conditionals\Front_End_Conditional;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Robots_Helper;
use Yoast\WP\SEO\Integrations\Integration_Interface;
use Yoast\WP\SEO\Schema\Application\Configuration\Schema_Configuration;

/**
 * Frontend_Inspector class
 */
class Frontend_Inspector implements Integration_Interface {

	/**
	 * The identifier used for the frontend inspector submenu.
	 *
	 * @var string
	 */
	public const FRONTEND_INSPECTOR_SUBMENU_IDENTIFIER = 'wpseo-frontend-inspector';

	/**
	 * Holds the Robots_Helper.
	 *
	 * @var Robots_Helper
	 */
	protected $robots_helper;

	/**
	 * The options helper.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * The schema configuration.
	 *
	 * @var Schema_Configuration
	 */
	protected $schema_configuration;

	/**
	 * Constructs a Frontend_Inspector.
	 *
	 * @param Robots_Helper        $robots_helper        The Robots_Helper.
	 * @param Options_Helper       $options_helper       The options helper.
	 * @param Schema_Configuration $schema_configuration The schema configuration.
	 */
	public function __construct(
		Robots_Helper $robots_helper,
		Options_Helper $options_helper,
		Schema_Configuration $schema_configuration
	) {
		$this->robots_helper        = $robots_helper;
		$this->options_helper       = $options_helper;
		$this->schema_configuration = $schema_configuration;
	}

	/**
	 * {@inheritDoc}
	 */
	public static function get_conditionals() {
		return [ Front_End_Conditional::class ];
	}

	/**
	 * {@inheritDoc}
	 */
	public function register_hooks() {
		\add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ], 11 );
		\add_action( 'wpseo_add_adminbar_submenu', [ $this, 'add_frontend_inspector_submenu' ], 10, 2 );
	}

	/**
	 * Adds the frontend inspector submenu.
	 *
	 * @param WP_Admin_Bar $wp_admin_bar    The admin bar.
	 * @param string       $menu_identifier The menu identifier.
	 *
	 * @return void
	 */
	public function add_frontend_inspector_submenu( WP_Admin_Bar $wp_admin_bar, $menu_identifier ) {
		if ( ! \is_admin() ) {
			$menu_args = [
				'parent' => $menu_identifier,
				'id'     => self::FRONTEND_INSPECTOR_SUBMENU_IDENTIFIER,
				'title'  => \__( 'Front-end SEO inspector', 'wordpress-seo-premium' ),
				'href'   => '#wpseo-frontend-inspector',
				'meta'   => [
					'tabindex' => '0',
				],
			];
			$wp_admin_bar->add_menu( $menu_args );
		}
	}

	/**
	 * Enqueue the workouts app.
	 *
	 * @return void
	 */
	public function enqueue_assets() {
		if ( ! \is_admin_bar_showing() || ! $this->options_helper->get( 'enable_admin_bar_menu' ) ) {
			return;
		}

		// If the current user can't write posts, this is all of no use, so let's not output an admin menu.
		if ( ! \current_user_can( 'edit_posts' ) ) {
			return;
		}

		$analysis_seo         = new WPSEO_Metabox_Analysis_SEO();
		$analysis_readability = new WPSEO_Metabox_Analysis_Readability();
		$current_page_meta    = \YoastSEO()->meta->for_current_page();
		$indexable            = $current_page_meta->indexable;
		$page_type            = $current_page_meta->page_type;

		$is_seo_analysis_active         = $analysis_seo->is_enabled();
		$is_readability_analysis_active = $analysis_readability->is_enabled();
		$display_metabox                = true;

		switch ( $page_type ) {
			case 'Home_Page':
			case 'Post_Type_Archive':
			case 'Date_Archive':
			case 'Error_Page':
			case 'Fallback':
			case 'Search_Result_Page':
				break;
			case 'Static_Home_Page':
			case 'Static_Posts_Page':
			case 'Post_Type':
				$display_metabox = $this->options_helper->get( 'display-metabox-pt-' . $indexable->object_sub_type );
				break;
			case 'Term_Archive':
				$display_metabox = $this->options_helper->get( 'display-metabox-tax-' . $indexable->object_sub_type );
				break;
			case 'Author_Archive':
				$display_metabox = false;
				break;
		}

		if ( ! $display_metabox ) {
			$is_seo_analysis_active         = false;
			$is_readability_analysis_active = false;
		}

		\wp_enqueue_script( 'yoast-seo-premium-frontend-inspector' );
		\wp_localize_script(
			'yoast-seo-premium-frontend-inspector',
			'wpseoScriptData',
			[
				'frontendInspector' => [
					'isSchemaEnabled'                  => ! $this->schema_configuration->is_schema_disabled_programmatically(), // Even if schema is disabled via the settings, it's technically disabled programmatically.
					'isIndexable'                      => $this->robots_helper->is_indexable( $indexable ),
					'indexable'                        => [
						'is_robots_noindex'           => $indexable->is_robots_noindex,
						'primary_focus_keyword'       => $indexable->primary_focus_keyword,
						'primary_focus_keyword_score' => $indexable->primary_focus_keyword_score,
						'readability_score'           => $indexable->readability_score,
					],
					'contentAnalysisActive'            => $is_readability_analysis_active,
					'keywordAnalysisActive'            => $is_seo_analysis_active,
					'trackingConfig'                   => [
						'firstActionedOn' => $this->options_helper->get( 'frontend_inspector_first_actioned_on', '' ),
						'optionName'      => 'frontend_inspector_first_actioned_on',
					],
				],
			]
		);
	}
}
